<?php

/**
 * @package         Convert Forms
 * @version         1.0.5 Free
 * 
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2016 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
*/

defined('_JEXEC') or die('Restricted access');
require_once __DIR__ . "/smarttags.php";
require_once JPATH_PLUGINS . "/system/nrframework/helpers/cache.php";
jimport('joomla.log.log');

class ConvertFormsHelper
{
    /**
     *  Returns Upgrade URL with Google Analytics Parameters
     *
     *  @return  string
     */
    public static function getUpgradeURL()
    {
        return NRFrameworkFunctions::getUTMURL("http://www.tassos.gr/joomla-extensions/convert-forms/");
    }

    /**
     *  Writes the Not Found Items message
     *
     *  @param   string  $view 
     *
     *  @return  string
     */
    public static function noItemsFound($view = "leads")
    {
        $html[] = '<span style="font-size:16px; position:relative; top:2px;" class="icon-smiley-sad-2"></span>';
        $html[] = JText::sprintf("COM_CONVERTFORMS_NO_RESULTS_FOUND", JText::_("COM_CONVERTFORMS_" . $view));

        return implode(" ", $html);
    }

    /**
     *  Get Latest Leads
     *
     *  @param   integer  $limit  The max number of lead records
     *
     *  @return  object
     */
    public static function getLatestLeads($limit = 10)
    {
        $model = JModelLegacy::getInstance('Conversions', 'ConvertFormsModel', array('ignore_request' => true));
        $model->setState('list.limit', 10);

        return $model->getItems();
    }

    /**
     *  Get Visitor ID
     *
     *  @return  string
     */
    public static function getVisitorID()
    {
        if (!@include_once(JPATH_PLUGINS . "/system/nrframework/helpers/visitor.php"))
        {
            return;
        }

        return NRVisitor::getInstance()->getToken();
    }

    /**
     *  Returns campaigns list visitor is subscribed to
     *  If the user is logged in, we try to get the campaigns by user's ID
     *  Otherwise, the visitor cookie ID will be used instead
     *
     *  @return  array  List of campaign IDs
     */
    public static function getVisitorCampaigns()
    {
        $db    = JFactory::getDBO();
        $query = $db->getQuery(true);
        $user  = JFactory::getUser();

        $query
            ->select('campaign_id')
            ->from('#__convertforms_conversions')
            ->where('state = 1')
            ->group('campaign_id');

        // Get leads by user id if visitor is logged in
        if ($user->id)
        {
            $query->where('user_id = ' . (int) $user->id);
        } else 
        {
            // Get leads by Visitor Cookie
            if (!$visitorID = self::getVisitorID())
            {
                return false;
            }

            $query->where('visitor_id = ' . $db->q($visitorID));
        }

        $db->setQuery($query);
        return $db->loadColumn();
    }

    /**
     *  Returns an array with all available campaigns
     *
     *  @return  array
     */
    public static function getCampaigns()
    {
        JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_convertforms/models');

        $model = JModelLegacy::getInstance('Campaigns', 'ConvertFormsModel', array('ignore_request' => true));
        $model->setState('filter.state', 1);

        return $model->getItems();
    }

    /**
     *  Logs messages to log file
     *
     *  @param   string  $msg   The message
     *  @param   object  $type  The log type
     *
     *  @return  void
     */
    public static function log($msg, $type = null)
    {
        $type = is_null($type) ? JLog::DEBUG : $type;
        JLog::add($msg, $type, 'com_convertforms');
    }

    /**
     *  Loads pre-made form template 
     *
     *  @param   string  $name  The template name
     *
     *  @return  object         
     */
    public static function loadFormTemplate($name)
    {
        $form = JPATH_ROOT . "/media/com_convertforms/templates/" . $name . ".cnvf";

        if (!JFile::exists($form))
        {
            return;
        }

        $content = file_get_contents($form);

        if (empty($content))
        {
            return;
        }

        $item = json_decode($content, true);
        $item = $item[0];

        $data = (object) array_merge((array) $item, (array) json_decode($item["params"]));
        $data->id = 0;
        $data->campaign = null;
        $data->fields = (array) $data->fields;

        return $data;
    }

    /**
     * Configure the Linkbar.
     *
     * @param   string  $vName  The name of the active view.
     *
     * @return  void
     */
    public static function addSubmenu($vName)
    {
        JHtmlSidebar::addEntry(
            JText::_('NR_DASHBOARD'),
            'index.php?option=com_convertforms',
            'convertforms'
        );
        JHtmlSidebar::addEntry(
            JText::_('COM_CONVERTFORMS_FORMS'),
            'index.php?option=com_convertforms&view=forms',
            'forms'
        );
        JHtmlSidebar::addEntry(
            JText::_('COM_CONVERTFORMS_CAMPAIGNS'),
            'index.php?option=com_convertforms&view=campaigns',
            'campaigns'
        );
        JHtmlSidebar::addEntry(
            JText::_('COM_CONVERTFORMS_CONVERSIONS'),
            'index.php?option=com_convertforms&view=conversions',
            'conversions'
        );
        JHtmlSidebar::addEntry(
            JText::_('COM_CONVERTFORMS_ADDONS'),
            'index.php?option=com_convertforms&view=addons',
            'addons'
        );
    }

    /**
     *  Returns permissions
     *
     *  @return  object
     */
    public static function getActions()
    {
        $user = JFactory::getUser();
        $result = new JObject;
        $assetName = 'com_convertforms';

        $actions = array(
            'core.admin', 'core.manage', 'core.create', 'core.edit', 'core.edit.state', 'core.delete'
        );

        foreach ($actions as $action)
        {
            $result->set($action, $user->authorise($action, $assetName));
        }

        return $result;
    }

    /**
     *  Returns a form object from database
     *
     *  @param   Integer  $id  The Form ID
     *
     *  @return  object       The Form object
     */
    public static function getForm($id)
    {
        if (!$id)
        {
            return;
        }

        // Get a db connection.
        $db = JFactory::getDbo();
        $query = $db->getQuery(true);
         
        $query->select("*")
            ->from($db->quoteName('#__convertforms'))
            ->where($db->quoteName('id') . ' = '. $id)
            ->where($db->quoteName('state') . ' = 1');
         
        $db->setQuery($query);

        return $db->loadAssoc();
    }

    /**
     *  Prepares a form object for rendering
     *
     *  @param   Object  $form  The form object
     *
     *  @return  array          The prepared form array
     */
    public static function prepareForm($item)
    {
        if (!$item || !isset($item["params"]))
        {
            return;
        }

        $item["id"] = isset($item["id"]) ? $item["id"] : 0;
        $classPrefix = "cf";

        // Replace variables
        $item["params"] = ConvertFormsSmartTags::replace($item["params"]);

        /* Decode Params */
        $params = new JRegistry($item["params"]);
        $item["params"] = $params;

        /* Box Classes */
        $boxClasses = array(
            "convertforms",
            $classPrefix,
            $classPrefix . "-" . $params->get("formlayout", "ver1col"),
            $classPrefix . "-" . $params->get("imgposition"),
            $classPrefix . "-" . $params->get("formposition"),
            $params->get("hideform", true) ? $classPrefix . "-success-hideform" : null,
            !$params->get("hidelabels", true) ? $classPrefix . "-hasLabels" : null,
            $params->get("centerform", false) ? $classPrefix . "-isCentered" : null,
            $params->get("classsuffix", null)
        );

        /* Box Styles */
        $boxStyle = array(
            "max-width:"        . ($params->get("autowidth", "auto") == "auto" ? "auto" : (int) $params->get("width", "500") . "px"),
            "background-color:" . $params->get("bgcolor", "transparent"),
            "border-style:"     . $params->get("borderstyle", "solid"),
            "border-width:"     . (int) $params->get("borderwidth", "2") . "px",
            "border-color:"     . $params->get("bordercolor", "#000"),
            "border-radius:"    . (int) $params->get("borderradius", "0") . "px",
            "padding:"          . (int) $params->get("padding", "20") . "px",
            (!is_null($params->get("font"))) ? "font-family:" . $params->get("font") : null
        );

        // Background Image
        if ($params->get("bgimage", false))
        {
            $imgurl = intval($params->get("bgimage")) == 1 ? JURI::root() . $params->get("bgfile") : $params->get("bgurl");
            $bgImage = array(
                "background-image: url(". $imgurl .")",
                "background-repeat:"   . strtolower($params->get("bgrepeat")),
                "background-size:"     . strtolower($params->get("bgsize")),
                "background-position:" . strtolower($params->get("bgposition"))
            );

            $boxStyle = array_merge($bgImage, $boxStyle);
        }

        // Form HTML Attributes
        $item["boxattributes"] = implode(" ",
            array(
                'id="' . $classPrefix . '_' . $item["id"] . '"',
                'class="' . implode(" ", $boxClasses) . '"',
                'style="' . implode(";", $boxStyle) . '"'
            )
        );

        // Field Styles
        $fieldStyles = array(
            "text-align:"       . $params->get("inputalign", "left"),
            "color:"            . $params->get("inputcolor", "#888"),
            "background-color:" . $params->get("inputbg"),
            "border-color:"     . $params->get("inputbordercolor", "#ccc"),
            "border-radius:"    . (int) $params->get("inputborderradius", "0") . "px",
            "font-size:"        . (int) $params->get("inputfontsize", "13") . "px",
            "padding:"          . (int) $params->get("inputvpadding", "9") . "px " . (int) $params->get("inputhpadding", "10") . "px"
        ); 

        // Field Classes
        $fieldClasses = array(
            $classPrefix . "-input",
            $classPrefix . "-input-shadow-" . ($params->get("inputshadow", "false") ? "1" : "0")
        );

        // Fields Attributes
        $item["fieldattributes"] = implode(" ",
            array(
                'class=" ' . implode(" ", $fieldClasses) . '"',
                'style=" ' . implode(";", $fieldStyles) . '"'
            )
        );

        // Button Container Classes
        $item["buttonContainerClasses"] = implode(" ", array(
            ($params->get("formlayout") == "hor" ? $classPrefix . "-col-medium-" . $params->get("submitsize") : null)
        ));

        // Button Styles
        $buttonStyles = array(
            "border-radius:" . (int) $params->get("btnborderradius") . "px",
            "padding:"       . (int) $params->get("btntvpadding") . "px " . (int) $params->get("btnhpadding") . "px",
            "color:"         . $params->get("btntextcolor", "#fff"),
            "width:"         . ($params->get("btnalign") == "full" || $params->get("formlayout") == "hor" ? "100%" : "auto"),
            "font-size:"     . (int) $params->get("btnfontsize") . "px"
        );

        if (in_array($params->get("btnstyle", "flat"), array("gradient", "flat")))
        {
            $buttonStyles = array_merge($buttonStyles, array(
                "background-color:" . $params->get("btnbg", "#44A326"),
            ));

            $item["styles"][] = '
                #cf_'.$item["id"].' .cf-btn:after { 
                    border-radius: '.(int) $params->get("btnborderradius", "5") . "px".'
                }
            ';
        }

        if ($params->get("btnstyle", "flat") == "outline")
        {
            $buttonStyles = array_merge($buttonStyles, array(
                "border: solid 1px " . $params->get("btnbg"),
                "background: none"
            ));

            $item["styles"][] = '
                #cf_'.$item["id"].' .cf-btn:hover { 
                    background-color: '.$params->get("btnbg").' !important;
                    color: '.$params->get("btntexthovercolor", "#fff").' !important;
                }
            ';
        }

        // Field Classes
        $buttonClasses = array(
            $classPrefix . "-btn",
            $classPrefix . "-btn-style-"  . $params->get("btnstyle", "flat"),
            $classPrefix . "-btn-shadow-" . ($params->get("btnshadow", false) ? "1" : "0")
        );

        // Fields Attributes
        $item["buttonattributes"] = implode(" ",
            array(
                'class="' . implode(" ", $buttonClasses) . '"',
                'style="' . implode(";", $buttonStyles) . '"'
            )
        );

        // Main Image Checks
        $imageOption = $params->get("image");
        if ($imageOption == '1')
        {
            if (JFile::exists(JPATH_SITE . "/" . $params->get("imagefile")))
            {
                $item["image"] = JURI::root() . $params->get("imagefile");
            }
        }
        if ($imageOption == '2')
        {
            $item["image"] = $params->get("imageurl");
        }


        // Labels Styles
        $item["labelStyles"] = array(
            "color:"     . $params->get("labelscolor"),
            "font-size:" . (int) $params->get("labelsfontsize") . "px"
        );

        // Fields Container
        $item["fieldscontainerclasses"] = implode(" ", array(
            ($params->get("formlayout") == "hor" ? $classPrefix . "-col-medium-" . (16 - $params->get("submitsize")) : null)
        ));   

        // Image Container
        $item["imagecontainerclasses"] = implode(" ", array(
            (in_array($params->get("imgposition"), array("img-right", "img-left")) ? $classPrefix . "-col-medium-" . $params->get("imagesize", 6) : null),
        ));

        // Image
        $item["imagestyles"] = array(
            "width:" . ($params->get("imageautowidth", "auto") == "auto" ? "auto" : (int) $params->get("imagewidth", "500") . "px"),
            "left:". (int) $params->get("imagehposition", "0") . "px ",
            "top:". (int) $params->get("imagevposition", "0") . "px"
        );
        $item["imageclasses"] = array(
            ($params->get("hideimageonmobile", false) ? "cf-hide-mobile" : "")
        );

        // Form Container
        $item["formclasses"] = array(
            (in_array($params->get("formposition"), array("form-left", "form-right")) ? $classPrefix . "-col-large-" . $params->get("formsize", 6) : null),
        );
        $item["formstyles"] = array(
            "background-color:" . $params->get("formbgcolor", "none")
        );

        // Content
        $item["contentclasses"] = implode(" ", array(
            (in_array($params->get("formposition"), array("form-left", "form-right")) ? $classPrefix . "-col-large-" . (16 - $params->get("formsize", 6)) : null),
        ));

        // Text Container
        $item["textcontainerclasses"] = implode(" ", array(
            (in_array($params->get("imgposition"), array("img-right", "img-left")) ? $classPrefix . "-col-medium-" . (16 - $params->get("imagesize", 6)) : null),
        ));

        $item["hiddenfields"] = array();

        $textContent = trim($params->get("text", null));
        $footerContent = trim($params->get("footer", null));

        $item["textIsEmpty"]   = empty($textContent);
        $item["footerIsEmpty"] = empty($footerContent);
        $item["hascontent"]    = !$item["textIsEmpty"] || (bool) isset($item["image"]) ?  1 : 0;

        // Prepare Fields
        $item["fields"] = array();
        foreach ($params->get("fields") as $key => $field)
        {
            if (!isset($field->name) || empty($field->name))
            {
                $field->name = "unknown".$key;
            }

            switch ($field->type)
            {
                case 'hidden':
                    $item["hiddenfields"][] = $field;
                    break;
                case 'dropdown':
                case 'checkbox':
                    self::prepareChoices($field);
                    $item["fields"][] = $field;
                    break;
                default:
                    $item["fields"][] = $field;
                    break;
            }
        }

        // Load custom fonts into the document
        include_once JPATH_PLUGINS . "/system/nrframework/helpers/fonts.php";
        NRFonts::loadFont($params->get("font"));

        // Finalizing Item
        $item["styles"][] = $params->get("customcss");

        unset($item["params"]["fields"]);
        unset($item["params"]["customcss"]);

        return $item;
    }

    public static function prepareChoices(&$field)
    {
        if (!isset($field->dropdownoptions) || !isset($field->dropdownoptions->choices))
        {
            return;
        }

        $choices = array();

        $hasPlaceholder = ($field->type == 'dropdown' && isset($field->placeholder) && !empty($field->placeholder));

        // Create a new array of valid only choices
        foreach ($field->dropdownoptions->choices as $key => $choiceValue)
        {
            if (!isset($choiceValue->label) || empty($choiceValue->label))
            {
                continue;
            }

            $label = trim($choiceValue->label);
            $value = empty($choiceValue->value) ? $label : $choiceValue->value;

            $choices[] = array(
                'label'    => $label,
                'value'    => $value,
                'selected' => (isset($choiceValue->default) && $choiceValue->default && !$hasPlaceholder) ? true : false
            );
        }

        // If it's a dropdown field and we have a placeholder available, add it to dropdown choices.
        if ($hasPlaceholder)
        {
            array_unshift($choices, array(
                'label'    => trim($field->placeholder),
                'value'    => '',
                'selected' => true,
                'disabled' => true
            ));
        }

        $field->dropdownoptions->choices = $choices;
    }

    /**
     *  Renders form by ID
     *
     *  @param   integer  $id  The form ID
     *
     *  @return  string        The form HTML
     */
    public static function renderFormById($id)
    {
        $hash = md5('convertform' . $id);

        if (NRCache::has($hash))
        {
            return NRCache::get($hash);
        }

        if (!$data = self::getForm((int) $id))
        {
            return;
        }

        self::loadassets(true);

        return NRCache::set($hash, self::renderForm($data));
    }

    /**
     *  Renders Form
     *
     *  @param   integer  $formid  The form id
     *
     *  @return  string            The form HTML
     */
    public static function renderForm($data)
    {
        $data = self::prepareForm($data);
        $layout = new JLayoutFile("form", null, array('debug' => 0, 'client' => 1, 'component' => 'com_convertforms'));
        return $layout->render($data);
    }

    /**
     *  Loads components media files
     *
     *  @param   boolean  $front
     *
     *  @return  void
     */
    public static function loadassets($frontend = false)
    {
        static $run;

        if ($run)
        {
            return;
        }

        $run = true;

        // Front-end media files
        if ($frontend)
        {
            $params = JComponentHelper::getParams('com_convertforms');

            if ($params->get("loadjQuery", true))
            {
                JHtml::_('jquery.framework');
            }

            $mediaVersioning = $params->get("mediaversioning", true);
            NRFrameworkFunctions::addMedia("convertforms.js", "com_convertforms", $mediaVersioning);

            if ($params->get("loadCSS", true))
            {
                NRFrameworkFunctions::addMedia("convertforms.css", "com_convertforms", $mediaVersioning);
            }

            JFactory::getDocument()->addScriptDeclaration('
                var ConvertFormsConfig = {
                    "baseurl" : "' . JURI::root() . '",
                    "debug"   : ' . $params->get("debug", "false") . '
                };
            ');

            return;
        }

        JHtml::_('jquery.framework');

        // Back-end Media Files
        NRFrameworkFunctions::addMedia(array("convertforms.sys.js", "convertforms.sys.css"), "com_convertforms");

        $version = NRFrameworkFunctions::getExtensionVersion("com_convertforms");
        // Make sure Novarain Fields CSS is loaded
        JFactory::getDocument()->addStylesheet(JURI::root(true) . "/plugins/system/nrframework/assets/css/fields.css?v=".$version);
    }

    /**
     *  Get Campaign Item by ID
     *
     *  @param   integer  $id  The campaign ID
     *
     *  @return  object
     */
    public static function getCampaign($id)
    {
        $model = JModelLegacy::getInstance('Campaign', 'ConvertFormsModel', array('ignore_request' => true));
        return $model->getItem($id);
    }
}