<?php

/**
 * @package         Convert Forms
 * @version         1.0.5 Free
 * 
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2016 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
*/

defined('_JEXEC') or die('Restricted access');

jimport('joomla.filesystem.file');
jimport('joomla.filesystem.folder');

/**
 *  Convert Forms Plugin
 */
class PlgSystemConvertForms extends JPlugin
{
    /**
     *  Application Object
     *
     *  @var  object
     */
    protected $app;

    /**
     *  Component's param object
     *
     *  @var  JRegistry
     */
    private $param;

    /**
     *  The loaded indicator of helper
     *
     *  @var  boolean
     */
    private $init;

    /**
     *  Log Object
     *
     *  @var  Object
     */ 
    private $log;

    /**
     *  AJAX Response
     *
     *  @var  stdClass
     */
    private $response;

    /**
     *  Plugin constructor
     *
     *  @param  mixed   &$subject
     *  @param  array   $config
     */
    public function __construct(&$subject, $config = array())
    {
        $component = JComponentHelper::getComponent('com_convertforms', true);

        if (!$component->enabled)
        {   
            return;
        }

        $this->param = $component->params;

        // Declare extension logger
        JLog::addLogger(
            array('text_file' => 'com_convertforms.php'),
            JLog::ALL, 
            array('com_convertforms')
        );

        // execute parent constructor
        parent::__construct($subject, $config);
    }

    /**
     *  onAfterRoute Event
     */
    public function onAfterRoute()
    {
        // Get Helper
        if (!$this->getHelper())
        {
            return;
        }
    }

    /**
     *  Handles the content preparation event fired by Joomla!
     *
     *  @param   mixed     $context     Unused in this plugin.
     *  @param   stdClass  $article     An object containing the article being processed.
     *  @param   mixed     $params      Unused in this plugin.
     *  @param   int       $limitstart  Unused in this plugin.
     *
     *  @return  bool
     */
    public function onContentPrepare($context, &$article, &$params, $limitstart = 0)
    {
        // Get Helper
        if (!$this->getHelper())
        {
            return true;
        }

        // Check whether the plugin should process or not
        if (JString::strpos($article->text, 'convertforms') === false)
        {
            return true;
        }

        // Search for this tag in the content
        $regex = "#{convertforms (.*?)}#s";
        $article->text = preg_replace_callback($regex, array('self', 'process'), $article->text);
    }

    /**
     *  Callback to preg_replace_callback in the onContentPrepare event handler of this plugin.
     *
     *  @param   array   $match  A match to the {convertforms} plugin tag
     *
     *  @return  string  The processed result
     */
    private static function process($match)
    {
        if (!isset($match[1]))
        {
            return;
        }

        return ConvertFormsHelper::renderFormById($match[1]);
    }

    /**
     *  Listens to AJAX requests on ?option=com_ajax&format=raw&plugin=convertforms
     *  Method aborts on invalid token or task
     *
     *  @return void
     */
    function onAjaxConvertForms()
    {
        // Check if we have a valid task
        $task = $this->app->input->get('task', null);

        if (is_null($task))
        {
            die('Invalid task');
        }

        // An access token is required on all requests except on the API task which 
        // has a native authentication method through an API Key
        if ($task != "api")
        {
            JSession::checkToken('request') or jexit(JText::_('JINVALID_TOKEN'));
        }

        // Cool access granted.
        $componentPath = JPATH_ADMINISTRATOR . '/components/com_convertforms/';
        JModelLegacy::addIncludePath($componentPath . 'models');
        JTable::addIncludePath($componentPath . 'tables');

        // Load component helper
        require_once $componentPath . '/helpers/convertforms.php';

        // Load component language file
        NRFrameworkFunctions::loadLanguage("com_convertforms");

        // Check if we have a valid method task
        $taskMethod = "ajaxTask".$task;

        if (!method_exists($this, $taskMethod))
        {
            die('Task not found');
        }

        // Success! Let's call the method.
        $this->response = new stdClass();

        try
        {
           $this->$taskMethod();
        }
        catch (Exception $e)
        {
            $this->response->error = $e->getMessage();
        }

        echo json_encode($this->response, JSON_PRETTY_PRINT);
    }

    /**
     *  Handles Submit AJAX task
     *  
     *  @return  void
     */
    private function ajaxTaskSubmit()
    {
        // Try to create a new conversion
        $model = JModelLegacy::getInstance('Conversion', 'ConvertFormsModel', array('ignore_request' => true));
        $conversion = $model->createConversion();

        $this->response->success = true;
        $this->response->task = $conversion->form->onsuccess;

        switch ($conversion->form->onsuccess)
        {
            case "msg":
                $this->response->value     = $conversion->form->successmsg;
                $this->response->hideform  = $conversion->form->hideform;
                $this->response->resetform = $conversion->form->resetform;
                break;
            case "url":
                $this->response->value     = $conversion->form->successurl;
                $this->response->passdata  = $conversion->form->passdata;
                break;
        }

        $this->response->value = ConvertFormsSmartTags::replace($this->response->value, $conversion);
    }
    
    # PRO-START
    /**
     *  AJAX Method to retrieve service account lists
     *  
     *  Listens to requests on ?option=com_ajax&format=raw&plugin=convertforms&task=lists
     *  Required arguments: service=[SERVICENAME]&key=[APIKEY/ACCESSTOKEN]
     *
     *  @return void
     */
    private function ajaxTaskLists()
    {
        // Check if we have a valid API key / Access Token
        $key = $this->app->input->get('key', null);

        if (is_null($key) || empty($key))
        {
            die('Invalid Key');
        }

        // Check if we have a valid service name passed
        $service = $this->app->input->get('service', null);

        if (is_null($service) || empty($service))
        {
            die('Invalid Service');
        }

        // Yeah! We have a service! Dispatcher call the plugins please!
        JPluginHelper::importPlugin('convertforms');

        $dispatcher = JEventDispatcher::getInstance();
        $lists = $dispatcher->trigger('onConvertFormsServiceLists', array($service, $key));

        if (is_array($lists[0]))
        {
            $this->response->lists = $lists[0];
        } else 
        {
            $this->response->error = $lists[0];
        }
    }

    /**
     *  API Task help us query ConvertForms tables and output the result as JSON
     *
     *  @return  void
     */
    private function ajaxTaskAPI()
    {
        // Run only if API is enabled
        $params = JComponentHelper::getParams('com_convertforms');

        if (!$params->get("api", false))
        {
            ConvertFormsHelper::log('JSON-API is disabled. Enable it through ConvertForms configuration page.');
            die();
        }

        require_once JPATH_ADMINISTRATOR . '/components/com_convertforms/helpers/api.php';

        $endpoint = $this->app->input->get('endpoint', 'forms');
        $apikey   = $this->app->input->get('api_key');

        $api = new ConvertFormsAPI($apikey);

        $this->response = $api->route($endpoint);

        JFactory::getDocument()->setMimeEncoding('application/json');
    }

    # PRO-END
    
    /**
     *  Map onContentAfterSave event to onConvertFormsConversionAfterSave
     *  
     *  Content is passed by reference, but after the save, so no changes will be saved.
     *  Method is called right after the content is saved.
     *
     *  @param   string  $context  The context of the content passed to the plugin (added in 1.6)
     *  @param   object  $article  A JTableContent object
     *  @param   bool    $isNew    If the content has just been created
     *
     *  @return  void
     */
    public function onContentAfterSave($context, $article, $isNew)
    {
        if ($context != "com_convertforms.conversion")
        {
            return;
        }

        JPluginHelper::importPlugin('convertforms');

        // Load item row
        $model = JModelLegacy::getInstance('Conversion', 'ConvertFormsModel', array('ignore_request' => true));
        if (!$conversion = $model->getItem($article->id))
        {
            return;
        }

        $dispatcher = JEventDispatcher::getInstance();
        $dispatcher->trigger('onConvertFormsConversionAfterSave', array($conversion, $model, $isNew));
    }

     /**
     *  Prepare form.
     *
     *  @param   JForm  $form  The form to be altered.
     *  @param   mixed  $data  The associated data for the form.
     *
     *  @return  boolean
     */
    public function onContentPrepareForm($form, $data)
    {
        // Return if we are in frontend
        if ($this->app->isSite())
        {
            return true;
        }

        // Check we have a form
        if (!($form instanceof JForm))
        {
            return false;
        }

        // Check we have a valid form context
        $validForms = array(
            "com_convertforms.campaign",
            "com_convertforms.form"
        );

        if (!in_array($form->getName(), $validForms))
        {
            return true;
        }

        // Load ConvertForms plugins
        JPluginHelper::importPlugin('convertforms');
        $dispatcher = JEventDispatcher::getInstance();
        $results = array();

        // Campaign Forms
        if ($form->getName() == "com_convertforms.campaign")
        {
            if (!isset($data->service) || !$service = $data->service)
            {
                return true;
            }
            
            $result = $dispatcher->trigger('onConvertFormsCampaignPrepareForm', array($form, $data, $service));
        }

        // Form Editing Page
        if ($form->getName() == "com_convertforms.form")
        {
            $result = $dispatcher->trigger('onConvertFormsFormPrepareForm', array($form, $data));
        }

        // Check for errors encountered while preparing the form.
        if (count($results) && in_array(false, $results, true))
        {
            // Get the last error.
            $error = $dispatcher->getError();
            if (!($error instanceof Exception))
            {
                throw new Exception($error);
            }
        }

        return true;
    }

    /**
     *  Loads the helper classes of plugin
     *
     *  @return  bool
     */
    private function getHelper()
    {
        // Return if is helper is already loaded
        if ($this->init)    
        {
            return true;
        }

        // Return if we are not in frontend
        if (!$this->app->isSite())
        {
            return false;
        }

        // Handle the component execution when the tmpl request paramter is overriden
        if (!$this->param->get("executeoutputoverride", false) && $this->app->input->get('tmpl', null, "cmd") != null)
        {
            return false;
        }

        // Handle the component execution when the format request paramter is overriden
        if (!$this->param->get("executeonformat", false) && $this->app->input->get('format', "html", "cmd") != "html")
        {
            return false;
        }

        // Check if Novarain Framework is installed
        if (!JFile::exists(JPATH_PLUGINS . '/system/nrframework/nrframework.php')) 
        {
            return false;
        }

        // Check if Novarain Framework is enabled
        $p = JPluginHelper::getPlugin('system', "nrframework");
        if (!isset($p->name))
        {
            return false;
        }
        
        // Load Novarain Framework Helpers
        require_once JPATH_PLUGINS . '/system/nrframework/helpers/functions.php';
        
        // Return if document type is Feed
        if (NRFrameworkFunctions::isFeed())
        {
            return false;
        }

        // Load component's helper file
        $componentHelper = JPATH_ADMINISTRATOR.'/components/com_convertforms/helpers/convertforms.php';
        if (!JFile::exists($componentHelper))
        {
            return false;
        }
        
        require_once $componentHelper;

        return ($this->init = true);
    }
}
