<?php

/**
 * @package         Google Structured Data Markup
 * @version         1.2 Free
 *
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2016 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

defined('_JEXEC') or die('Restricted access');

class plgSystemGSD extends JPlugin
{

	/**
	 *  Auto loads the plugin language file
	 *
	 *  @var  boolean
	 */
	protected $autoloadLanguage = true;

	/**
	 *  The loaded indicator of helper
	 *
	 *  @var  boolean
	 */
	protected $init;

	/**
	 *  Application Object
	 *
	 *  @var  object
	 */
	protected $app;

	/**
	 *  Making sure the Novarain Framework is loaded in the backend
	 */
	public function onAfterRoute()
	{
		if ($this->app->isAdmin() && !class_exists("NRFrameworkFunctions"))
		{
			require_once JPATH_PLUGINS . '/system/nrframework/helpers/functions.php';
		}
	}

	/**
	 *  Event to add custom script tags to the document
	 *
	 *  @return  boolean    True on success
	 */
	public function onBeforeRender()
	{

		// Load Helper
		if (!$this->getHelper())
		{
			return;
		}

		// Add Google Structured Markup
		$this->addGoogleStructuredMarkup();
	}

	/**
	 *  Adds Google Structured Markup in the document in JSON Format
	 */
	private function addGoogleStructuredMarkup()
	{
		// Get JSON markup for each available type
		$data = [
			$this->getJSONSiteName(),
			
			$this->getJSONBreadcrumbs(),
		];

		// Convert data array to string
		$markup = implode("\n", array_filter($data));

		// Return if markup is empty
		if (!$markup || empty($markup) || is_null($markup))
		{
			return;
		}

		// Add final markup to the document
		JFactory::getDocument()->addCustomTag('
            <!-- Start: ' . JText::_("GSD") . ' -->
            ' . $markup . '
            <!-- End: ' . JText::_("GSD") . ' -->
        ');
	}

	/**
	 *  Adds Breadcrumbs structured data markup
	 *  This is ideal for responsive websites that disable/hide breadcrumbs for mobile devices.
	 *  https://developers.google.com/structured-data/breadcrumbs
	 *
	 *  @return  string
	 */
	private function getJSONBreadcrumbs()
	{

		if (!$this->params->get("breadcrumbs_enabled", true) || $this->isFrontPage())
		{
			return;
		}

		$crumbs = $this->getCrumbs();

		if (!$crumbs)
		{
			return;
		}

		$crumbsData = [];

		foreach ($crumbs as $key => $value)
		{
			$crumbsData[] = '{
                "@type": "ListItem",
                "position": ' . ($key + 1) . ',
                "item": {
                    "@id":  "' . $value->link . '",
                    "name": "' . $value->name . '"
                }
            }';
		}

		$data = '
            <script type="application/ld+json">
            {
                "@context": "http://schema.org",
                "@type": "BreadcrumbList",
                "itemListElement": [' . implode(",", $crumbsData) . ']
            }
            </script>
        ';

		return $data;
	}

	/**
	 *  Returns Site Name Strucuted Data Markup
	 *  https://developers.google.com/structured-data/site-name
	 *
	 *  @return  string
	 */
	private function getJSONSiteName()
	{

		if (!$this->params->get("sitename_enabled", true) || !$this->isFrontPage())
		{
			return;
		}

		$siteData = [
			'"@context": "http://schema.org"',
			'"@type": "WebSite"',
			'"name": "' . $this->getSiteName() . '"',
			'"url": "' . $this->getSiteURL() . '"',
		];

		if ($this->params->get("sitename_name_alt"))
		{
			$siteData[] = '"alternateName": "' . $this->params->get("sitename_name_alt") . '"';
		}

		$data = '
            <script type="application/ld+json">
            {
                ' . implode(",\n", $siteData) . '
            }
            </script>
        ';

		return $data;
	}

	

	/**
	 *  Returns an array with crumbs
	 *
	 *  @return  array
	 */
	private function getCrumbs()
	{

		$pathway = $this->app->getPathway();
		$items   = $pathway->getPathWay();
		$menu    = $this->app->getMenu();
		$lang    = JFactory::getLanguage();
		$count   = count($items);

		// Look for the home menu
		if (JLanguageMultilang::isEnabled())
		{
			$home = $menu->getDefault($lang->getTag());
		}
		else
		{
			$home = $menu->getDefault();
		}

		if (!$count)
		{
			return false;
		}

		// We don't use $items here as it references JPathway properties directly
		$crumbs = [];

		for ($i = 0; $i < $count; $i++)
		{
			$crumbs[$i]       = new stdClass;
			$crumbs[$i]->name = stripslashes(htmlspecialchars($items[$i]->name, ENT_COMPAT, 'UTF-8'));
			$crumbs[$i]->link = JRoute::_($items[$i]->link, true, -1);
		}

		// Add Home item
		$item       = new stdClass;
		$item->name = htmlspecialchars($this->params->get('breadcrumbs_home', JText::_('GSD_BREADCRUMBS_HOME')));
		$item->link = JRoute::_('index.php?Itemid=' . $home->id, true, -1);
		array_unshift($crumbs, $item);

		return $crumbs;
	}

	/**
	 *  Get site Name
	 *
	 *  @return  string  Site URL
	 */
	private function getSiteName()
	{
		return $this->params->get("sitename_name", JFactory::getConfig()->get('sitename'));
	}

	/**
	 *  Get site URL
	 *
	 *  @return  string  Site URL
	 */
	private function getSiteURL()
	{
		return $this->params->get("sitename_url", JURI::base());
	}

	/**
	 *  Determine if the user is viewing the front page
	 *
	 *  @return  boolean
	 */
	private function isFrontPage()
	{
		$menu = $this->app->getMenu();
		$lang = JFactory::getLanguage()->getTag();
		return ($menu->getActive() == $menu->getDefault($lang));
	}

	/**
	 *  Loads Helper files
	 *
	 *  @return  bool
	 */
	private function getHelper()
	{
		// Return if is helper is already loaded
		if ($this->init)
		{
			return true;
		}

		// Return if we are not in frontend
		if (!$this->app->isSite())
		{
			return false;
		}

		// Only in HTML format
		if ($this->app->input->get('format', "html", "cmd") != "html")
		{
			return false;
		}

		// Check if Novarain Framework is installed
		jimport('joomla.filesystem.file');
		if (!JFile::exists(JPATH_PLUGINS . '/system/nrframework/nrframework.php'))
		{
			return false;
		}

		// Check if Novarain Framework is enabled
		$p = JPluginHelper::getPlugin('system', "nrframework");
		if (!isset($p->name))
		{
			return false;
		}

		// Load Novarain Framework Helper
		require_once JPATH_PLUGINS . '/system/nrframework/helpers/functions.php';

		// Return if current page is an XML page
		if (NRFrameworkFunctions::isFeed() || $this->app->input->getInt('print', 0))
		{
			return false;
		}

		return ($this->init = true);
	}
}
