<?php

/**
 * @package         Google Structured Data
 * @version         4.0.1 Free
 * 
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2018 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
*/

namespace GSD;

defined('_JEXEC') or die('Restricted access');

USE GSD\Json;
use GSD\Helper;
use GSD\MappingOptions;
use NRFramework\Cache;
use NRFramework\Assignments;
use Joomla\Registry\Registry;

/**
 *  Google Structured Data helper class
 */
class PluginBase extends \JPlugin
{
    /**
     *  Auto load the plugin language file
     *
     *  @var  boolean
     */
    protected $autoloadLanguage = true;

    /**
     *  Joomla Application Object
     *
     *  @var  object
     */
    protected $app;

    /**
     *  Joomla Database Object
     *
     *  @var  object
     */
    protected $db;

    /**
     *  Holds all available snippets for the current active page.
     *
     *  @var  array
     */
    protected $snippets;

    /**
     *  Indicates the query string parameter name that is used by the front-end component
     *
     *  @var  string
     */
    protected $thingRequestIDName = 'id';

    /**
     *  Indicates the request variable name used by plugin's assosiated component
     *
     *  @var  string
     */
    protected $thingRequestViewVar = 'view';

    /**
     *  Plugin constructor
     *
     *  @param  mixed   &$subject
     *  @param  array   $config
     */
    public function __construct(&$subject, $config = array())
    {
        // Load main language file
        \JFactory::getLanguage()->load('plg_system_gsd', JPATH_PLUGINS . '/system/gsd');

        // execute parent constructor
        parent::__construct($subject, $config);
    }

    /**
     *  Event triggered to gather all available plugins.
     *  Mostly used by the dropdowns in the backend.
     *
     *  @param   boolean  $mustBeInstalled  If enabled, the assosiated component must be installed
     *
     *  @return  array
     */
    public function onGSDGetType($mustBeInstalled = true)
    {
        if ($mustBeInstalled && !\NRFramework\Extension::isInstalled($this->_name))
        {
            return;
        }

        return array(
            'name'  => \JText::_('PLG_GSD_' . strtoupper($this->_name) . '_ALIAS'),
            'alias' => $this->_name
        );
    }

     /**
     *  Prepare form.
     *
     *  @param   JForm  $form  The form to be altered.
     *  @param   mixed  $data  The associated data for the form.
     *
     *  @return  boolean
     */
    public function onContentPrepareForm($form, $data)
    {
        // Make sure we are on the right context
        if ($this->app->isSite() || $form->getName() != 'com_gsd.item')
        {
            return;
        }

        if (is_null($data->plugin) || $data->plugin != $this->_name)
        {
            return;
        }

        // Load assignments xml file if it's available
        $assignmentsXML = JPATH_PLUGINS . '/gsd/' . $this->_name . '/form/assignments.xml'; 
        if (!\JFile::exists($assignmentsXML))
        {
            return;
        }

        $form->loadFile($assignmentsXML, false);
    }
    
    /**
     *  The event triggered before the JSON markup be appended to the document.
     *
     *  @param   array  &$data   The JSON snippets to be appended to the document
     *
     *  @return  void
     */
    public function onGSDBeforeRender(&$data)
    {
        // Quick filtering on component check
        if (!$this->passContext())
        {
            return;
        }

        // Let's check if the plugin supports the current component's view.
        if (!$payload = $this->getPayload())
        {
            return;
        }

        // Now, let's see if we have valid snippets for the active page. If not abort.
        if (!$this->snippets = $this->getSnippets())
        {
            $this->log('No valid items found');
            return;
        }

        // Prepare snippets
        foreach ($this->snippets as $snippet)
        {
            // Here, the payload must be merged with the snippet data
            $jsonData = $this->preparePayload($snippet, $payload);

            // Create JSON
            $jsonClass = new Json($jsonData);
            $json = $jsonClass->generate();

            // Add json back to main data object
            $data[] = $json;
        }
    }

    /**
     *  Validate context to decide whether the plugin should run or not.
     *
     *  @return   bool
     */
    protected function passContext()
    {
        return Helper::getComponentAlias() == $this->_name;
    }

    /**
     *  Get Item's ID
     *
     *  @return  string
     */
    protected function getThingID()
    {
        return $this->app->input->getInt($this->thingRequestIDName);
    }

    /**
     *  Get component's items and validate conditions
     *
     *  @return  Mixed   Null if no items found, The valid items array on success
     */
    protected function getSnippets()
    {
        \JModelLegacy::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_gsd/models');

        $model = \JModelLegacy::getInstance('Items', 'GSDModel', ['ignore_request' => true]);
        $model->setState('filter.plugin', $this->_name);
        $model->setState('filter.state', 1);

        if (!$rows = $model->getItems())
        {
            return;
        }

        // Check publishing assignments for each item
        foreach ($rows as $key => $row)
        {
            if (!isset($row->assignments) || !is_object($row->assignments))
            {
                continue;
            }

            // Prepare assignments
            $assignmentsFound = [];

            foreach ($row->assignments as $alias => $assignment)
            {
                if ($assignment->assignment_state == '0')
                {
                    continue;
                }

                // Remove unwanted assignments added by Free Pro code blocks
                if (strpos($alias, '@'))
                {
                    continue;
                }

                // Comply with the new conditions requirements
                $condition = (object) [
                    'alias'  => $alias,
                    'value'  => $assignment->selection,
                    'params' => isset($assignment->params) ? $assignment->params : [],
                    'assignment_state' => $assignment->assignment_state
                ];

                $assignmentsFound[] = [$condition];
            }

            // Validate assignments
            if (!$pass = (new Assignments())->passAll($assignmentsFound))
            {
                $this->log('Item #' . $row->id . ' does not pass the conditions check');
                unset($rows[$key]);
            }
        }

        $items = array_map(function($row)
        {
            $contentType = $row->contenttype;

            $s = new Registry($row->{$contentType});
            $s->set('contentType', $contentType);

            return $s;
        }, $rows);

        return $items;
    }

    /**
     *  Asks for data from the child plugin based on the active view name
     *
     *  @return  Registry  The payload Registry
     */
    private function getPayload()
    {
        $view   = $this->getView();
        $method = 'view' . ucfirst($view);

        if (!$view || !method_exists($this, $method))
        {
            $this->log('View ' . $view . ' is not supported');
            return;
        }

        // Yeah. Let's call the method. 
        $payload = $this->$method();

        // We need a valid array
        if (!is_array($payload))
        {  
            $this->log('Invalid Payload Array');
            return;
        }

        // Convert payload to Registry object and return it
        return new Registry($payload);
    }

    /**
     *  Prepares the payload to be used in the JSON class
     *
     *  @return  string
     */
    private function preparePayload($snippet, $payload)
    {   
        MappingOptions::prepare($snippet);

        // Create a new combined object by merging the snippet data into the payload
        // Note: In order to produce a valid merged object, payload's array keys should match the field names
        // as declared in the form's XML file.
        $p = clone $payload;
        $s = $p->merge($snippet, false);

        // Replace Smart Tags - This can be implemented with a Plugin
        $s = MappingOptions::replace($s, $payload);

        // Shorthand for the content type
        $contentType    = $snippet['contentType'];
        $prepareContent = Helper::getParams()->get('preparecontent', false);
        $descCharsLimit = Helper::getParams()->get('desclimit', 300);

        // Prepare common data
        $commonData = array(
            'contentType'   => $contentType,
            'title'         => Helper::makeTextSafe($s['headline'], $prepareContent, 110),
            'description'   => Helper::makeTextSafe($s['description'], $prepareContent, $descCharsLimit),
            'image'         => Helper::absURL($s->get('image')),

            // Author / Publisher
            'authorName'    => $s['author'],
            'publisherName' => Helper::getSiteName(),
            'publisherLogo' => Helper::getSiteLogo(),

            // Rating
            'ratingValue'   => $s['rating_value'],
            'reviewCount'   => $s['review_count'],
            'bestRating'    => $s['bestRating'],
            'worstRating'   => $s['worstRating'],

            // Dates
            'datePublished' => Helper::date($s['publish_up']),
            'dateCreated'   => Helper::date($s['created']),
            'dateModified'  => Helper::date($s['modified']),

            // Site based
            'url'           => \JURI::current(),
            'siteurl'       => Helper::getSiteURL(),
            'sitename'      => Helper::getSiteName()
        );

        // Prepare snippet data
        $data = array();
        switch ($contentType)
        {
            case 'article':
                // Article's headline should not exceed the 110 characters
                $commonData['title'] = mb_substr($commonData['title'], 0, 110);
                break;
            case 'product':
                $data = array(
                    "offerPrice"   => $s['offerPrice'],
                    "brand"        => $s['brand'],
                    "sku"          => $s['sku'],
                    "currency"     => $s['currency'],
                    "condition"    => $s['offerItemCondition'],
                    "availability" => $s['offerAvailability']
                );
                break;
            case 'event':
                $data = array(
                    "type"      => $s['type'],
                    "startdate" => Helper::date($s['startDate']),
                    "enddate"   => Helper::date($s['endDate']),
                    "location"  => array("name" => $s['locationName'], "address" => $s['locationAddress']),
                    "performer" => array("name" => $s['performerName'], "type" => $s['performerType']),
                    "status"    => $s['status'],
                    "offer"     => array(
                        "availability"   => $s['offerAvailability'], 
                        "startDateTime"  => Helper::date($s['offerStartDate']),
                        "price"          => $s['offerPrice'],
                        "currency"       => $s['offerCurrency'],
                        "inventoryLevel" => $s['offerInventoryLevel']
                    )
                );

                break;
            case 'recipe':
                $data = array(
                    "prepTime"      => $s['prepTime'] ? "PT" . $s['prepTime'] . "M" : null,
                    "cookTime"      => $s['cookTime'] ? "PT" . $s['cookTime'] . "M" : null,
                    "totalTime"     => $s['totalTime'] ? "PT" . $s['totalTime'] . "M" : null,
                    "calories"      => $s['calories'],
                    "yield"         => $s['yield'],
                    "ingredient"    => Helper::makeArrayFromNewLine($s['ingredient']),
                    "instructions"  => Helper::makeArrayFromNewLine($s['instructions']),
                    'category'      => $s['category'],
                    'cuisine'       => $s['cuisine'],
                    'video'         => $s['video'],
                    'keywords'      => $s['keywords'],
                );
                break;
            case 'review':
                $data = array(
                    'itemReviewedType' => $s['itemReviewedType'],
                    'itemReviewedURL'  => $s['itemReviewedURL'],
                    'itemReviewedPublishedDate' => $s['item_reviewed_published_date'],
                    'movie_director'   => $s['item_reviewed_movie_director'],
                    'book_author'      => $s['item_reviewed_book_author'],
                    'book_author_url'  => $s['item_reviewed_book_author_url'],
                    'book_isbn'        => $s['item_reviewed_book_isbn'],
                    'address'          => $s['address'],
                    'priceRange'       => $s['priceRange'],
                    'telephone'        => $s['telephone'],
                    'language_code'    => explode('-', \JFactory::getLanguage()->getTag())[0]
                );
                break;
            case 'factcheck':
                switch ($s['factcheckRating']) {
                    // there is no textual representation for zero (0)
                    case '1':
                        $textRating = 'False';
                        break;
                    case '2':
                        $textRating = 'Mostly false';
                        break;
                    case '3':
                        $textRating = 'Half true';
                        break;
                    case '4':
                        $textRating = 'Mostly true';
                        break;
                    case '5':
                        $textRating = 'True';
                        break;
                    default:
                        $textRating = 'Hard to categorize';
                        break;
                }

                $data = array(
                    "factcheckURL"          => ($s['multiple']) ? $commonData['url'] . $s['anchorName'] : $commonData['url'],
                    "claimAuthorType"       => $s['claimAuthorType'],
                    "claimAuthorName"       => $s['claimAuthorName'],
                    "claimURL"              => $s['claimURL'],
                    "claimDatePublished"    => $s['claimDatePublished'],
                    "factcheckRating"       => $s['factcheckRating'],
                    "bestFactcheckRating"   => ($s['factcheckRating'] != '-1') ? '5' : '-1',
                    "worstFactcheckRating"  => ($s['factcheckRating'] != '-1') ? '1' : '-1',
                    "alternateName"         => $textRating
                );
                break;
            case 'video':
                $data = array(
                    "contentUrl" => $s['contentUrl']
                );
                break;
            case 'custom_code':
                $data = array(
                    'custom' => $s['custom_code']
                );
                break;
            case 'jobposting':
                $data = [
                    'hiring_oprganization_name' => $s['hiring_oprganization_name'],
                    'hiring_oprganization_url' => $s['hiring_oprganization_url'],
                    'country' => $s['addressCountry'],
                    'address' => $s['streetAddress'],
                    'region' => $s['region'],
                    'locality' => $s['locality'],
                    'postal_code' => $s['postal_code'],
                    'industry' => $s['industry'],
                    'education' => $s['educationRequirements'],
                    'valid_through'  => $s['valid_through'],
                    'employmenttype' => $s['employmenttype'],
                    'salary' => (strpos($s['salary'], '-') === false ? $s['salary'] : explode('-', $s['salary'])),
                    'salary_currency' => $s['currency'],
                    'salary_unit' => $s['salary_unit']
                ];
                break;
        }

        return array_merge($data, $commonData);
    }

    /**
     *  Get View Name
     *
     *  @return  string  Return the current executed view in the front-end
     */
    protected function getView()
    {
        return $this->app->input->get($this->thingRequestViewVar);
    }

    /**
     *  Log messages
     *
     *  @param   string  $message  The message to log
     *
     *  @return  void
     */
    protected function log($message)
    {
        Helper::log(\JText::_('PLG_GSD_' . $this->_name . '_ALIAS') . ' - ' . $message);
    }
}

?>